/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ccvs.core.client;

import java.util.Date;

import org.eclipse.team.internal.ccvs.core.CVSTag;

/**
 * The "cvs rlog..." command
 */
public class RLog extends RemoteCommand {
	
	/*** Local options: specific to rlog ***/
	public static final LocalOption NO_TAGS = new LocalOption("-N", null); //$NON-NLS-1$
	
	/**
	 * Makes a -r option for rlog. Here are the currently supported options:
	 * 
	 * tag1		tag2		result
	 * ==== ==== =================================
	 * date  	date		date<date (all revisions between date and later)
	 * tag		tag		tag:tag (all revisions between tag and tag, must be on same branch)
 	 * branch date 	>date (all revisions of date or later)
	 * branch tag		tag: (all revisions from tag to the end of branchs tip)
	 * 
	 * Valid for: rlog
	 */
	public static LocalOption makeTagOption(CVSTag tag1, CVSTag tag2) {
		int type1 = tag1.getType();
		int type2 = tag2.getType();
		
		if(type1 == type2) {
			switch (type1) {
				case CVSTag.HEAD:
				case CVSTag.BRANCH:
					// A range of branches - all revisions on all the branches in that range.
				case CVSTag.VERSION:
					// Revisions from tag1 to tag2 (they must be on the same branch).
					return new LocalOption("-r" + tag1.getName() + ":" + tag2.getName(), null); //$NON-NLS-1$ //$NON-NLS-2$
				case CVSTag.DATE:
					// Selects revisions created between DATE1 and DATE2. If DATE1 is after DATE2, use > instead; otherwise, no log messages are retrieved.
					Date date1 = tag1.asDate();
					Date date2 = tag2.asDate();
					String operator = "<"; //$NON-NLS-1$
					if(date1.compareTo(date2) > 0) {
						operator = ">"; //$NON-NLS-1$
					}
					return new LocalOption("-d'" + tag1.getName() + "'"+ operator + "'" + tag2.getName() + "'", null); //$NON-NLS-1$ //$NON-NLS-2$
				default:
					// Unknow tag type!!!
					throw new IllegalArgumentException();
			}
		}
		
		if((type1 == CVSTag.BRANCH || type1 == CVSTag.HEAD) && type2 == CVSTag.DATE) {
			return new LocalOption("-d", ">" + tag2.getName()); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		if((type1 == CVSTag.BRANCH || type1 == CVSTag.HEAD) && type2 == CVSTag.VERSION) {
			return new LocalOption("-r" + tag2.getName() + ":", null); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		// defaults
		switch (type1) {
			case CVSTag.HEAD:
			case CVSTag.BRANCH:
				// All revisions on this branch
			case CVSTag.VERSION:
				// revisions in this tag
				return new LocalOption("-r" + tag1.getName(), null); //$NON-NLS-1$ //$NON-NLS-2$
			case CVSTag.DATE:
				// Revisions at this date tag
				return new LocalOption("-d", tag1.getName()); //$NON-NLS-1$ //$NON-NLS-2$
			default:
				// Unknow tag type!!!
				throw new IllegalArgumentException();
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.team.internal.ccvs.core.client.Request#getRequestId()
	 */
	protected String getRequestId() {
		return "rlog"; //$NON-NLS-1$
	}
}
