/**********************************************************************
Copyright (c) 2007 Thomas Marti and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.awt.geom.Ellipse2D;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;

/**
 * Mapping for java.awt.geom.Ellipse2D.Float, maps the x, y, width and height values to float-precision 
 * datastore fields.
 */
public class Ellipse2dFloatMapping extends SingleFieldMultiMapping
{

    private static final Ellipse2D.Float sampleValue = new Ellipse2D.Float(0, 0, 1, 1);

	/* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#initialize()
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
		super.initialize(dba, fmd, container, clr);

        addDatastoreField(ClassNameConstants.FLOAT); // X
        addDatastoreField(ClassNameConstants.FLOAT); // Y
        addDatastoreField(ClassNameConstants.FLOAT); // Width
        addDatastoreField(ClassNameConstants.FLOAT); // Height
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return Ellipse2D.Float.class;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getSampleValue()
     */
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return sampleValue;
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object, int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
    	Ellipse2D.Float ellipse = (Ellipse2D.Float)value;
        if (ellipse == null)
        {
    		for (int i = 0; i < exprIndex.length; i++) 
    		{
    			getDataStoreMapping(i).setObject(preparedStatement, exprIndex[i], null);					
			}
        }
        else
        {
            getDataStoreMapping(0).setFloat(preparedStatement,exprIndex[0],ellipse.x);
            getDataStoreMapping(1).setFloat(preparedStatement,exprIndex[1],ellipse.y);
            getDataStoreMapping(2).setFloat(preparedStatement,exprIndex[2],ellipse.width);
            getDataStoreMapping(3).setFloat(preparedStatement,exprIndex[3],ellipse.height);
        }
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        // Check for null entries
        if (getDataStoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
        {
            return null;
        }

        float x = getDataStoreMapping(0).getFloat(resultSet,exprIndex[0]); 
        float y = getDataStoreMapping(1).getFloat(resultSet,exprIndex[1]); 
        float width  = getDataStoreMapping(2).getFloat(resultSet,exprIndex[2]); 
        float height = getDataStoreMapping(3).getFloat(resultSet,exprIndex[3]);
        return new Ellipse2D.Float(x, y, width, height);
    }
}