/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Text;
using System.Text.RegularExpressions;
using System.Collections;

namespace PostgreSql.Data.PgSqlClient.DbSchema
{
	#region Structures

	internal struct PgColumn
	{
		public string ColumnName;
		public string ColumnAlias;
		public string WhereColumnName;
	}

	internal struct PgTableJoin
	{
		public string JoinType;
		public string RightTable;
		public string Expression;
	}

	internal struct PgPrivilege
	{
		public string			User;
		public MatchCollection	Privileges;
	}

	#endregion

	internal abstract class PgAbstractDbSchema : IDbSchema
	{
		#region Fields

		private ArrayList	restrictionColumns;
		private ArrayList	dataColumns;
		private ArrayList	tables;
		private ArrayList	joins;
		private ArrayList	orderByColumns;
		private ArrayList	whereFilters;

		private string		tableName;
		
		#endregion

		#region Properties

		public ArrayList RestrictionColumns
		{
			get { return restrictionColumns; }
		}

		#endregion

		#region Constructors

		public PgAbstractDbSchema()
		{
			restrictionColumns	= new ArrayList();
			dataColumns			= new ArrayList();
			tables				= new ArrayList();
			joins				= new ArrayList();
			orderByColumns		= new ArrayList();
			whereFilters		= new ArrayList();

			AddTables();
			AddRestrictionColumns();
			AddDataColumns();
			AddJoins();
			AddOrderByColumns();
			AddWhereFilters();
		}

		public PgAbstractDbSchema(string tableName) : this()
		{
			this.tableName = tableName;
		}

		#endregion

		#region Abstract Methods

		public abstract void AddTables();
		public abstract void AddRestrictionColumns();
		public abstract void AddDataColumns();
		public abstract void AddJoins();
		public abstract void AddOrderByColumns();
		public abstract void AddWhereFilters();
		public abstract object[] ParseRestrictions(object[] restrictions);

		#endregion

		#region Add Methods

		public void AddTable(string tableName)
		{
			tables.Add(tableName);
		}

		public void AddRestrictionColumn(string columnName, string columnAlias, string whereColumnName)
		{
			PgColumn column = new PgColumn();

			column.ColumnName	= columnName;
			column.ColumnAlias	= columnAlias;
			if (whereColumnName != null)
			{
				column.WhereColumnName = whereColumnName;
			}
			else
			{
				column.WhereColumnName = columnName;
			}


			restrictionColumns.Add(column);
		}

		public void AddDataColumn(string columnName, string columnAlias)
		{
			PgColumn column = new PgColumn();

			column.ColumnName	= columnName;
			column.ColumnAlias	= columnAlias;

			dataColumns.Add(column);
		}

		public void AddJoin(string joinType, string rightTable, string expression)
		{
			PgTableJoin join = new PgTableJoin();

			join.JoinType	= joinType;
			join.RightTable	= rightTable;
			join.Expression	= expression;			

			joins.Add(join);
		}

		public void AddOrderBy(string column)
		{
			orderByColumns.Add(column);
		}

		public void AddWhereFilter(string filter)
		{
			whereFilters.Add(filter);
		}

		#endregion

		#region Methods

		public virtual DataTable GetDbSchemaTable(PgConnection connection, object[] restrictions)
		{
			restrictions = ParseRestrictions(restrictions);

			DataSet			dataSet = null;
			PgDataAdapter	adapter = null;
			PgCommand		command = new PgCommand();
			
			try
			{
				command.Connection	= connection;
				command.CommandText = GetCommandText(restrictions);
				if (connection.ActiveTransaction != null &&
					!connection.ActiveTransaction.IsUpdated)
				{
					command.Transaction = connection.ActiveTransaction;
				}

				adapter = new PgDataAdapter(command);
				dataSet = new DataSet(tableName);

				adapter.Fill(dataSet, tableName);
			}
			catch (PgException pgex)
			{
				throw pgex;
			}
			catch (Exception ex)
			{
				throw new PgException(ex.Message);
			}
			finally
			{
				command.Dispose();
				adapter.Dispose();
			}

			return dataSet.Tables[tableName];
		}

		public string GetCommandText(object[] restrictions)
		{
			StringBuilder sql = new StringBuilder();

			// Add restriction columns
			sql.Append("SELECT ");
			foreach (PgColumn column in restrictionColumns)
			{
				sql.AppendFormat("{0} AS {1}", column.ColumnName, column.ColumnAlias);					
				if ((restrictionColumns.IndexOf(column) + 1) < restrictionColumns.Count)
				{
					sql.Append(", ");
				}
			}

			// Add DataColumns
			if (restrictionColumns.Count > 0 && dataColumns.Count > 0)
			{
				sql.Append(", ");
			}
			foreach (PgColumn column in dataColumns)
			{
				sql.AppendFormat("{0} AS {1}", column.ColumnName, column.ColumnAlias);					
				if ((dataColumns.IndexOf(column) + 1) < dataColumns.Count)
				{
					sql.Append(", ");
				}
			}

			// Add tables
			sql.Append(" FROM ");
			foreach (string table in tables)
			{
				sql.Append(table);
				if ((tables.IndexOf(table) + 1) < tables.Count)
				{
					sql.Append(", ");
				}
			}

			if (joins.Count != 0)
			{
				foreach (PgTableJoin join in joins)
				{
					sql.AppendFormat(" {0} {1} ON {2}", 
						join.JoinType,
						join.RightTable,
						join.Expression);
				}
			}

			// Add restrictions
			StringBuilder whereFilter = new StringBuilder();

			if (restrictions != null && restrictions.Length > 0)
			{
				for (int i = 0; i < restrictions.Length; i++)
				{
					if (restrictions[i] != null)
					{
						if (whereFilter.Length > 0)
						{
							whereFilter.Append(" AND ");
						}
						whereFilter.AppendFormat("{0} = '{1}'", 
							((PgColumn)restrictionColumns[i]).WhereColumnName, 
							restrictions[i]);
					}
				}
			}

			if (whereFilters != null && whereFilters.Count > 0)
			{
				foreach (string condition in whereFilters)
				{
					if (whereFilter.Length > 0)
					{
						whereFilter.Append(" AND ");
					}
					whereFilter.Append(condition);
				}
			}

			if (whereFilter.Length > 0)
			{
				sql.AppendFormat(" WHERE {0}", whereFilter);
			}

			// Add Order By
			if (orderByColumns.Count > 0)
			{
				sql.Append(" ORDER BY ");

				foreach (string columnName in orderByColumns)
				{
					sql.Append(columnName);
					
					if ((orderByColumns.IndexOf(columnName) + 1) < orderByColumns.Count)
					{
						sql.Append(", ");
					}
				}				
			}

			return sql.ToString();
		}

		#endregion

		#region Protected Methods

		protected void AddPrivilegesColumns(DataTable table)
		{
			table.Columns.Add("INSERT", Type.GetType("System.Boolean"));
			table.Columns["INSERT"].DefaultValue = false;

			table.Columns.Add("INSERT_WITH_GRANT", Type.GetType("System.Boolean"));
			table.Columns["INSERT_WITH_GRANT"].DefaultValue = false;

			table.Columns.Add("SELECT", Type.GetType("System.Boolean"));
			table.Columns["SELECT"].DefaultValue = false;

			table.Columns.Add("SELECT_WITH_GRANT", Type.GetType("System.Boolean"));
			table.Columns["SELECT_WITH_GRANT"].DefaultValue = false;

			table.Columns.Add("UPDATE", Type.GetType("System.Boolean"));
			table.Columns["UPDATE"].DefaultValue = false;

			table.Columns.Add("UPDATE_WITH_GRANT", Type.GetType("System.Boolean"));
			table.Columns["UPDATE_WITH_GRANT"].DefaultValue = false;

			table.Columns.Add("DELETE", Type.GetType("System.Boolean"));
			table.Columns["DELETE"].DefaultValue = false;

			table.Columns.Add("DELETE_WITH_GRANT", Type.GetType("System.Boolean"));
			table.Columns["DELETE_WITH_GRANT"].DefaultValue = false;

			table.Columns.Add("RULE", Type.GetType("System.Boolean"));
			table.Columns["RULE"].DefaultValue = false;

			table.Columns.Add("RULE_WITH_GRANT", Type.GetType("System.Boolean"));
			table.Columns["RULE_WITH_GRANT"].DefaultValue = false;

			table.Columns.Add("REFERENCES", Type.GetType("System.Boolean"));
			table.Columns["REFERENCES"].DefaultValue = false;

			table.Columns.Add("REFERENCES_WITH_GRANT", Type.GetType("System.Boolean"));
			table.Columns["REFERENCES_WITH_GRANT"].DefaultValue = false;

			table.Columns.Add("TRIGGER", Type.GetType("System.Boolean"));
			table.Columns["TRIGGER"].DefaultValue = false;

			table.Columns.Add("TRIGGER_WITH_GRANT", Type.GetType("System.Boolean"));
			table.Columns["TRIGGER_WITH_GRANT"].DefaultValue = false;
		}

		protected PgPrivilege[] DecodePrivileges(string[] acl)
		{
			Regex			search = new Regex(@"((a|r|w|d|R|x|t)\*?)");
			PgPrivilege[]	priv = new PgPrivilege[acl.Length];

			for (int i = 0; i < acl.Length; i++)
			{				
				priv[i].User = acl[i].Split('=')[0];
				if (priv[i].User.Length == 0)
				{
					priv[i].User = "PUBLIC";
				}
				string aclitem	= acl[i].Split('=')[1];
				aclitem = aclitem.Split('/')[0];

				priv[i].Privileges = search.Matches(aclitem);
			}

			return priv;
		}

		protected void FillPrivileges(DataRow row, MatchCollection privileges)
		{
			foreach (Match privilege in privileges)
			{
				switch (privilege.Value)
				{
					case "a":
					case "a*":
						row["INSERT"] = true;
						if (privilege.Value.IndexOf("*") != -1)
						{
							row["INSERT_WITH_GRANT"] = true;
						}
						break;
						
					case "r":
					case "r*":
						row["SELECT"] = true;
						if (privilege.Value.IndexOf("*") != -1)
						{
							row["SELECT_WITH_GRANT"] = true;
						}
						break;

					case "w":
					case "w*":
						row["UPDATE"] = true;
						if (privilege.Value.IndexOf("*") != -1)
						{
							row["UPDATE_WITH_GRANT"] = true;
						}
						break;

					case "d":
					case "d*":
						row["DELETE"] = true;
						if (privilege.Value.IndexOf("*") != -1)
						{
							row["DELETE_WITH_GRANT"] = false;
						}
						break;

					case "R":
					case "R*":
						row["RULE"] = true;
						if (privilege.Value.IndexOf("*") != -1)
						{
							row["RULE_WITH_GRANT"] = true;
						}
						break;

					case "x":
					case "x*":
						row["REFERENCES"] = true;
						if (privilege.Value.IndexOf("*") != -1)
						{
							row["REFERENCES_WITH_GRANT"] = true;
						}
						break;

					case "t":
					case "t*":
						row["TRIGGER"] = true;
						if (privilege.Value.IndexOf("*") != -1)
						{
							row["TRIGGER_WITH_GRANT"] = true;
						}
						break;
				}
			}
		}

		#endregion
	}
}
