import React, { useCallback, useEffect, useImperativeHandle, useState } from "react";
import { v4 as uuidv4 } from "uuid";
import { Button } from "@patternfly/react-core/dist/js/components/Button";
import { Grid, GridItem } from "@patternfly/react-core/dist/js/layouts/Grid";
import { Stack, StackItem } from "@patternfly/react-core/dist/js/layouts/Stack";
import { TextInput } from "@patternfly/react-core/dist/js/components/TextInput";
import PlusCircleIcon from "@patternfly/react-icons/dist/esm/icons/plus-circle-icon";
import TrashIcon from "@patternfly/react-icons/dist/esm/icons/trash-icon";
import { componentOuiaProps } from "@kie-tools/runtime-tools-components/dist/ouiaTools";
const CloudEventCustomHeadersEditor = React.forwardRef(({ ouiaId, ouiaSafe }, forwardedRef) => {
    const [headers, setHeaders] = useState([]);
    const [isNewHeader, setIsNewHeader] = useState(false);
    useEffect(() => {
        setIsNewHeader(false);
    }, [isNewHeader]);
    useImperativeHandle(forwardedRef, () => {
        return {
            reset() {
                setHeaders([]);
                setIsNewHeader(false);
            },
            getCustomHeaders() {
                const result = {};
                headers
                    .filter((entry) => entry.key && entry.value)
                    .forEach((entry) => {
                    result[entry.key] = entry.value;
                });
                return result;
            },
        };
    }, [headers]);
    const addNewHeader = useCallback(() => {
        const headersCopy = [...headers];
        headersCopy.push({
            uuid: uuidv4(),
            key: "",
            value: "",
        });
        setHeaders(headersCopy);
        setIsNewHeader(true);
    }, [headers]);
    const deleteHeader = useCallback((index) => {
        const headersCopy = [...headers];
        headersCopy.splice(index, 1);
        setHeaders(headersCopy);
    }, [headers]);
    const updateHeaderKey = useCallback((index, value) => {
        const headersCopy = [...headers];
        headersCopy[index].key = value;
        setHeaders(headersCopy);
    }, [headers]);
    const updateHeaderValue = useCallback((index, value) => {
        const headersCopy = [...headers];
        headersCopy[index].value = value;
        setHeaders(headersCopy);
    }, [headers]);
    return (React.createElement("div", Object.assign({}, componentOuiaProps(ouiaId, "custom-headers-editor", ouiaSafe)),
        React.createElement(Stack, { hasGutter: true },
            React.createElement(StackItem, null,
                React.createElement(Button, { key: "add-header-button", variant: "link", isInline: true, icon: React.createElement(PlusCircleIcon, null), onClick: () => addNewHeader() }, "Add Header")),
            headers.length > 0 && (React.createElement(StackItem, null,
                React.createElement(Grid, Object.assign({}, componentOuiaProps((ouiaId ? ouiaId : "custom-headers-editor") + "-grid", "custom-headers-editor", true)),
                    React.createElement(GridItem, { span: 4, key: "headers-grid-col-header" },
                        React.createElement("h4", null, "Header Name")),
                    React.createElement(GridItem, { span: 8, key: "headers-grid-col-value" },
                        React.createElement("h4", null, "Value")),
                    headers.map((header, index) => {
                        return (React.createElement(React.Fragment, { key: `headers-grid-row-${header.uuid}` },
                            React.createElement(GridItem, { span: 4, key: `header-key-${header.uuid}` },
                                React.createElement(TextInput, { id: `header-key-${index}-input`, value: header.key, onChange: (value) => updateHeaderKey(index, value), autoFocus: isNewHeader && index === headers.length - 1 })),
                            React.createElement(GridItem, { span: 7, key: `header-value-${header.uuid}` },
                                React.createElement(TextInput, { id: `header-value-${index}-input`, value: header.value, onChange: (value) => updateHeaderValue(index, value) })),
                            React.createElement(GridItem, { span: 1, key: `header-delete-${header.uuid}` },
                                React.createElement(Button, { variant: "plain", "aria-label": "delete", key: `header-delete-${index}-button`, onClick: () => deleteHeader(index) },
                                    React.createElement(TrashIcon, null)))));
                    })))))));
});
export default CloudEventCustomHeadersEditor;
//# sourceMappingURL=CloudEventCustomHeadersEditor.js.map