import { EditorTheme } from "@kie-tools-core/editor/dist/api";
import { Button } from "@patternfly/react-core/dist/js/components/Button";
import { Tooltip } from "@patternfly/react-core/dist/js/components/Tooltip";
import { UndoIcon } from "@patternfly/react-icons/dist/js/icons/undo-icon";
import { SaveIcon } from "@patternfly/react-icons/dist/js/icons/save-icon";
import { RedoIcon } from "@patternfly/react-icons/dist/js/icons/redo-icon";
import { PlayIcon } from "@patternfly/react-icons/dist/js/icons/play-icon";
import cloneDeep from "lodash/cloneDeep";
import React, { useEffect, useImperativeHandle, useMemo, useRef, useState } from "react";
import { useFormDetailsContext } from "../contexts/FormDetailsContext";
import "../styles.css";
import { FormEditorEditorController } from "./FormEditorController";
function FormEditorControl(props) {
    return (React.createElement(Tooltip, { content: React.createElement("div", null, props.toolTipText) },
        React.createElement(Button, { onClick: props.onClick, variant: "control", "aria-label": props.ariaLabel }, props.icon)));
}
export const FormEditor = React.forwardRef(({ code, formType, formContent, setFormContent, saveFormContent, isSource = false, isConfig = false }, forwardedRef) => {
    const [content, setContent] = useState("");
    const appContext = useFormDetailsContext();
    const container = useRef(null);
    const formLanguage = useMemo(() => {
        if (isSource && formType) {
            switch (formType.toLowerCase()) {
                case "tsx":
                    return "typescript";
                case "html":
                    return "html";
            }
        }
        else if (isConfig) {
            return "json";
        }
    }, [formType, isSource, isConfig]);
    const controller = useMemo(() => {
        return new FormEditorEditorController(code, (args) => setContent(args.content), formLanguage, false);
    }, [code, formLanguage]);
    useEffect(() => {
        if (container.current) {
            controller.show(container.current, EditorTheme.LIGHT);
        }
        return () => {
            controller.dispose();
        };
    }, [controller]);
    useImperativeHandle(forwardedRef, () => controller, [controller]);
    const onExecuteCode = () => {
        const tempContent = cloneDeep(formContent);
        const value = content;
        if (Object.keys(formContent)[0].length > 0 && isSource) {
            tempContent.source = value;
        }
        else {
            tempContent.configuration["resources"] = JSON.parse(value);
        }
        const newFormContent = Object.assign(Object.assign({}, formContent), tempContent);
        appContext.updateContent(newFormContent);
        setFormContent(newFormContent);
    };
    const onSaveForm = () => {
        saveFormContent(formContent);
    };
    const onUndoChanges = () => {
        controller.undo();
    };
    const onRedoChanges = () => {
        controller.redo();
    };
    const customControl = (React.createElement(React.Fragment, null,
        React.createElement(FormEditorControl, { icon: React.createElement(PlayIcon, null), ariaLabel: "Execute form", toolTipText: "Execute form", onClick: onExecuteCode }),
        React.createElement(FormEditorControl, { icon: React.createElement(UndoIcon, null), ariaLabel: "Undo changes", toolTipText: "Undo changes", onClick: onUndoChanges }),
        React.createElement(FormEditorControl, { icon: React.createElement(RedoIcon, null), ariaLabel: "Redo changes", toolTipText: "Redo changes", onClick: onRedoChanges }),
        React.createElement(FormEditorControl, { icon: React.createElement(SaveIcon, null), ariaLabel: "Save form", toolTipText: "Save form", onClick: () => onSaveForm() })));
    return (React.createElement(React.Fragment, null,
        React.createElement("div", null, customControl),
        React.createElement("div", { className: "kogito-form-editor-container", ref: container, style: { height: "700px" } })));
});
export default FormEditor;
//# sourceMappingURL=FormEditor.js.map