/*
 * Decompiled with CFR 0.152.
 */
package org.apache.knox.gateway.services.token.impl;

import java.lang.management.ManagementFactory;
import java.time.Instant;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.management.InstanceAlreadyExistsException;
import javax.management.MBeanRegistrationException;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import org.apache.knox.gateway.config.GatewayConfig;
import org.apache.knox.gateway.i18n.messages.MessagesFactory;
import org.apache.knox.gateway.services.ServiceLifecycleException;
import org.apache.knox.gateway.services.security.token.TokenStateService;
import org.apache.knox.gateway.services.security.token.TokenUtils;
import org.apache.knox.gateway.services.security.token.UnknownTokenException;
import org.apache.knox.gateway.services.security.token.impl.JWT;
import org.apache.knox.gateway.services.security.token.impl.JWTToken;
import org.apache.knox.gateway.services.token.TokenStateServiceStatistics;
import org.apache.knox.gateway.services.token.impl.TokenStateServiceMessages;

public class DefaultTokenStateService
implements TokenStateService {
    protected static final long DEFAULT_RENEWAL_INTERVAL = TimeUnit.HOURS.toMillis(24L);
    protected static final int MAX_RENEWALS = 7;
    protected static final long DEFAULT_MAX_LIFETIME = 7L * DEFAULT_RENEWAL_INTERVAL;
    protected static final TokenStateServiceMessages log = (TokenStateServiceMessages)MessagesFactory.get(TokenStateServiceMessages.class);
    private final Map<String, Long> tokenExpirations = new ConcurrentHashMap<String, Long>();
    private final Map<String, Long> maxTokenLifetimes = new ConcurrentHashMap<String, Long>();
    private long tokenEvictionInterval;
    private long tokenEvictionGracePeriod;
    protected boolean permissiveValidationEnabled;
    private final ScheduledExecutorService evictionScheduler = Executors.newScheduledThreadPool(1);
    protected TokenStateServiceStatistics tokenStateServiceStatistics;

    public void init(GatewayConfig config, Map<String, String> options) throws ServiceLifecycleException {
        this.tokenEvictionInterval = config.getKnoxTokenEvictionInterval();
        this.tokenEvictionGracePeriod = config.getKnoxTokenEvictionGracePeriod();
        this.permissiveValidationEnabled = config.isKnoxTokenPermissiveValidationEnabled();
        if (config.isMetricsEnabled() && config.isJmxMetricsReportingEnabled()) {
            try {
                this.tokenStateServiceStatistics = new TokenStateServiceStatistics();
                ObjectName objectName = ObjectName.getInstance("metrics:type=Statistics,name=TokenStateService");
                ManagementFactory.getPlatformMBeanServer().registerMBean(this.tokenStateServiceStatistics, objectName);
            }
            catch (InstanceAlreadyExistsException | MBeanRegistrationException | MalformedObjectNameException | NotCompliantMBeanException e) {
                throw new ServiceLifecycleException("Could not register token state service MBean", (Exception)e);
            }
        }
    }

    public void start() throws ServiceLifecycleException {
        if (this.tokenEvictionInterval > 0L) {
            this.evictionScheduler.scheduleAtFixedRate(this::evictExpiredTokens, this.tokenEvictionInterval, this.tokenEvictionInterval, TimeUnit.SECONDS);
        }
    }

    public void stop() throws ServiceLifecycleException {
        this.evictionScheduler.shutdown();
    }

    public long getDefaultRenewInterval() {
        return DEFAULT_RENEWAL_INTERVAL;
    }

    public long getDefaultMaxLifetimeDuration() {
        return DEFAULT_MAX_LIFETIME;
    }

    public void addToken(JWTToken token, long issueTime) {
        if (token == null) {
            throw new IllegalArgumentException("Token cannot be null.");
        }
        this.addToken(TokenUtils.getTokenId((JWT)token), issueTime, token.getExpiresDate().getTime());
    }

    public void addToken(String tokenId, long issueTime, long expiration) {
        this.addToken(tokenId, issueTime, expiration, this.getDefaultMaxLifetimeDuration());
    }

    public void addToken(String tokenId, long issueTime, long expiration, long maxLifetimeDuration) {
        this.validateTokenIdentifier(tokenId);
        this.tokenExpirations.put(tokenId, expiration);
        this.setMaxLifetime(tokenId, issueTime, maxLifetimeDuration);
        log.addedToken(tokenId, this.getTimestampDisplay(expiration));
        if (this.tokenStateServiceStatistics != null) {
            this.tokenStateServiceStatistics.addToken();
        }
    }

    public long getTokenExpiration(JWT token) throws UnknownTokenException {
        long expiration;
        block3: {
            expiration = -1L;
            try {
                expiration = this.getTokenExpiration(TokenUtils.getTokenId((JWT)token));
            }
            catch (UnknownTokenException e) {
                String exp;
                if (this.permissiveValidationEnabled && (exp = token.getExpires()) != null) {
                    log.permissiveTokenHandling(TokenUtils.getTokenId((JWT)token), e.getMessage());
                    expiration = Long.parseLong(exp);
                }
                if (expiration != -1L) break block3;
                throw e;
            }
        }
        return expiration;
    }

    public long getTokenExpiration(String tokenId) throws UnknownTokenException {
        return this.getTokenExpiration(tokenId, true);
    }

    public long getTokenExpiration(String tokenId, boolean validate) throws UnknownTokenException {
        Long expiration;
        if (validate) {
            this.validateToken(tokenId);
        }
        if ((expiration = this.tokenExpirations.get(tokenId)) == null) {
            throw new UnknownTokenException(tokenId);
        }
        return expiration;
    }

    public long renewToken(JWTToken token) throws UnknownTokenException {
        return this.renewToken(token, DEFAULT_RENEWAL_INTERVAL);
    }

    public long renewToken(JWTToken token, long renewInterval) throws UnknownTokenException {
        if (token == null) {
            throw new IllegalArgumentException("Token cannot be null.");
        }
        return this.renewToken(TokenUtils.getTokenId((JWT)token), renewInterval);
    }

    public long renewToken(String tokenId) throws UnknownTokenException {
        return this.renewToken(tokenId, DEFAULT_RENEWAL_INTERVAL);
    }

    public long renewToken(String tokenId, long renewInterval) throws UnknownTokenException {
        long expiration;
        this.validateToken(tokenId);
        if (this.hasRemainingRenewals(tokenId, renewInterval)) {
            expiration = System.currentTimeMillis() + renewInterval;
            this.updateExpiration(tokenId, expiration);
            log.renewedToken(tokenId, this.getTimestampDisplay(expiration));
            if (this.tokenStateServiceStatistics != null) {
                this.tokenStateServiceStatistics.renewToken();
            }
        } else {
            log.renewalLimitExceeded(tokenId);
            throw new IllegalArgumentException("The renewal limit for the token has been exceeded");
        }
        return expiration;
    }

    public void revokeToken(JWTToken token) throws UnknownTokenException {
        if (token == null) {
            throw new IllegalArgumentException("Token cannot be null.");
        }
        this.revokeToken(TokenUtils.getTokenId((JWT)token));
    }

    public void revokeToken(String tokenId) throws UnknownTokenException {
        this.removeToken(tokenId);
        log.revokedToken(tokenId);
    }

    public boolean isExpired(JWTToken token) throws UnknownTokenException {
        return this.getTokenExpiration((JWT)token) <= System.currentTimeMillis();
    }

    protected void setMaxLifetime(String token, long parsedMaxLifeTime) {
        this.maxTokenLifetimes.put(token, parsedMaxLifeTime);
    }

    protected void setMaxLifetime(String token, long issueTime, long maxLifetimeDuration) {
        this.maxTokenLifetimes.put(token, issueTime + maxLifetimeDuration);
    }

    protected boolean isUnknown(String token) {
        return !this.tokenExpirations.containsKey(token);
    }

    protected void updateExpiration(String tokenId, long expiration) {
        this.tokenExpirations.put(tokenId, expiration);
    }

    protected void removeToken(String tokenId) throws UnknownTokenException {
        this.validateToken(tokenId);
        this.removeTokens(Collections.singleton(tokenId));
    }

    protected void removeTokens(Set<String> tokenIds) throws UnknownTokenException {
        this.removeTokenState(tokenIds);
    }

    private void removeTokenState(Set<String> tokenIds) {
        this.tokenExpirations.keySet().removeAll(tokenIds);
        this.maxTokenLifetimes.keySet().removeAll(tokenIds);
        log.removedTokenState(String.join((CharSequence)", ", tokenIds));
    }

    protected boolean hasRemainingRenewals(String tokenId, long renewInterval) {
        return System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30L) + renewInterval < this.getMaxLifetime(tokenId);
    }

    protected long getMaxLifetime(String tokenId) {
        return this.maxTokenLifetimes.getOrDefault(tokenId, 0L);
    }

    private void validateTokenIdentifier(String tokenId) {
        if (tokenId == null || tokenId.isEmpty()) {
            throw new IllegalArgumentException("Token identifier cannot be null or empty.");
        }
    }

    protected void validateToken(String tokenId) throws IllegalArgumentException, UnknownTokenException {
        this.validateTokenIdentifier(tokenId);
        if (this.isUnknown(tokenId)) {
            log.unknownToken(tokenId);
            throw new UnknownTokenException(tokenId);
        }
    }

    private String getTimestampDisplay(long timestamp) {
        return Instant.ofEpochMilli(timestamp).toString();
    }

    protected void evictExpiredTokens() {
        if (this.readyForEviction()) {
            HashSet<String> tokensToEvict = new HashSet<String>();
            for (String tokenId : this.getTokenIds()) {
                try {
                    if (!this.needsEviction(tokenId)) continue;
                    log.evictToken(tokenId);
                    tokensToEvict.add(tokenId);
                }
                catch (Exception e) {
                    log.failedExpiredTokenEviction(tokenId, e);
                }
            }
            if (!tokensToEvict.isEmpty()) {
                try {
                    this.removeTokens(tokensToEvict);
                }
                catch (UnknownTokenException e) {
                    log.failedExpiredTokenEviction((Exception)((Object)e));
                }
            }
        } else {
            log.skipEviction();
        }
    }

    protected boolean readyForEviction() {
        return true;
    }

    protected boolean needsEviction(String tokenId) throws UnknownTokenException {
        long expirationWithGrace = this.getTokenExpiration(tokenId, false) + TimeUnit.SECONDS.toMillis(this.tokenEvictionGracePeriod);
        return expirationWithGrace <= System.currentTimeMillis();
    }

    private List<String> getTokenIds() {
        return this.tokenExpirations.keySet().stream().collect(Collectors.toList());
    }
}

