/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common;

import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jrobin.core.RrdDb;
import org.jrobin.core.RrdDbPool;
import org.jrobin.core.RrdDef;
import org.jrobin.core.RrdException;
import org.jrobin.core.Util;
import org.jrobin.graph.RrdGraph;
import org.jrobin.graph.RrdGraphDef;
import org.videolan.vlma.common.adapters.IVlAdapter;
import org.videolan.vlma.common.exceptions.InvalidServerNameException;

/**
 * A streaming server.
 * 
 * @author SylV
 * @version 1.0
 * 
 */
public class VlServer implements Serializable {

    protected static final Logger logger = Logger.getLogger(VlServer.class);

    protected String name;

    protected InetAddress ip;

    protected Map<String, IVlAdapter> adapters;

    protected boolean isUp;

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#isUp()
     */
    public boolean isUp() {
        return isUp;
    }

    /**
     * Returns the server ability to receive orders.
     * 
     * @return true if the server is ready, false otherwise
     */
    public boolean getUp() {
        return isUp;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#setUp(boolean)
     */
    public void setUp(boolean isUp) {
        this.isUp = isUp;
    }

    private void validateServerName(String name) {
        if (name.contains(File.separator)) throw new InvalidServerNameException("Server name musn't contain '" + File.separator + "'.");
    }
    
    /**
     * A server constructor.
     * 
     * @param name
     *            the server name
     * @param ip
     *            the server IP address
     */
    public VlServer(String name, InetAddress ip) {
        this.name = name;
        validateServerName(name);
        this.ip = ip;
        adapters = new HashMap<String, IVlAdapter>();
        setUp(true);
    }

    /**
     * Default constructor.
     * 
     */
    public VlServer() {
        this.name = "";
        this.ip = null;
    }

    /**
     * Gives the server name.
     * 
     * @return the server name
     */
    public String getName() {
        return name;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#setName(java.lang.String)
     */
    public void setName(String name) {
        this.name = name;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#getIp()
     */
    public InetAddress getIp() {
        return ip;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#setIp(java.net.InetAddress)
     */
    public void setIp(InetAddress ip) {
        this.ip = ip;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#getAdapters()
     */
    public Map<String, IVlAdapter> getAdapters() {
        return adapters;
    }

    protected double currentCpuLoad;

    protected long currentTrafficIn;

    protected long currentTrafficOut;

    protected double currentVlcCpu;

    protected double currentVlcMem;

    private File getRrdDir() {
        File basePath = new File(Configuration.getInstance().getWorkDir());
        File rrdDir = new File(basePath, "rrd");
        if (!rrdDir.exists()) rrdDir.mkdir();
        return rrdDir;
    }
    
    /**
     * Creates the RRD file if needed and returns its path.
     * 
     * @return the RRD file path
     */
    protected String createRrdFileIfNecessary() {
        File rrdR = new File(getRrdDir(), getName() + ".rrd");
        if (!rrdR.exists()) {
            logger
                    .log(Level.WARN, "Création du fichier de RRD de "
                            + getName());
            try {
                RrdDef rrdDef = new RrdDef(rrdR.getPath());
                rrdDef.setStartTime(Util.getTime());
                rrdDef.addDatasource("cpu_load", "GAUGE", 300, 0, Double.NaN);
                rrdDef.addDatasource("traffic_in", "COUNTER", 300, 0,
                        Double.NaN);
                rrdDef.addDatasource("traffic_out", "COUNTER", 300, 0,
                        Double.NaN);
                rrdDef.addDatasource("vlc_cpu", "GAUGE", 300, 0, Double.NaN);
                rrdDef.addDatasource("vlc_mem", "GAUGE", 300, 0, Double.NaN);
                rrdDef.addArchive("AVERAGE", 0.5, 1, 4000);
                rrdDef.addArchive("AVERAGE", 0.5, 6, 4000);
                rrdDef.addArchive("AVERAGE", 0.5, 24, 4000);
                rrdDef.addArchive("AVERAGE", 0.5, 288, 4000);
                RrdDbPool rrdPool = RrdDbPool.getInstance();
                RrdDb rrdDb = rrdPool.requestRrdDb(rrdDef);
                rrdPool.release(rrdDb);
            } catch (RrdException e) {
                logger.error("Error while creating RRD file of "
                        + rrdR.getPath(), e);
            } catch (IOException e) {
                logger.error("Error while creating RRD file of "
                        + rrdR.getPath(), e);
            }
        }
        return rrdR.getPath();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#updateSnmpData()
     */
    public void updateSnmpData() {
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#updateRrdGraph()
     */
    public void updateRrdGraph() {
        RrdGraph rrdGraph;
        long startTime, endTime = Util.getTime();
        startTime = endTime - 3600 * 24;

        try {
            File rrdDir = getRrdDir();
            
            RrdGraphDef graphDef = new RrdGraphDef();
            graphDef.datasource("cpu_load", createRrdFileIfNecessary(),
                    "cpu_load", "AVERAGE");
            graphDef.area("cpu_load", Color.RED, "CPU load@L");
            graphDef.line("cpu_load", Color.BLUE, "CPU load@L", 3);
            graphDef.gprint("cpu_load", "AVERAGE", "Average CPU load: @3@r");
            graphDef.setLowerLimit(0);
            graphDef.setTimePeriod(startTime, endTime);
            graphDef.setTitle("Load");
            rrdGraph = new RrdGraph(graphDef);
            File rrdGraphFile = new File(rrdDir, getName() + "-cpu_load.png");
            rrdGraph.saveAsPNG(rrdGraphFile.getAbsolutePath());

            graphDef = new RrdGraphDef();
            graphDef.datasource("traffic_out", createRrdFileIfNecessary(),
                    "traffic_out", "AVERAGE");
            graphDef.line("traffic_out", Color.BLUE, "Outgoing traffic@L", 3);
            graphDef.setLowerLimit(0);
            graphDef.setTimePeriod(startTime, endTime);
            graphDef.setTitle("Débit sortant sur eth0 sur un jour");
            rrdGraph = new RrdGraph(graphDef);
            rrdGraphFile = new File(rrdDir, getName() + "-traffic_out.png");
            rrdGraph.saveAsPNG(rrdGraphFile.getAbsolutePath());

            graphDef = new RrdGraphDef();
            graphDef.datasource("vlc_cpu", createRrdFileIfNecessary(),
                    "vlc_cpu", "AVERAGE");
            graphDef.datasource("vlc_mem", createRrdFileIfNecessary(),
                    "vlc_mem", "AVERAGE");
            graphDef.line("vlc_cpu", Color.RED, "CPU usage for VLC");
            graphDef.line("vlc_mem", Color.BLUE, "Memory usage for VLC");
            graphDef.setLowerLimit(0);
            graphDef.setTimePeriod(startTime, endTime);
            graphDef.setTitle("Resources used for VLC");
            rrdGraph = new RrdGraph(graphDef);
            rrdGraphFile = new File(rrdDir, getName() + "-vlc.png");
            rrdGraph.saveAsPNG(rrdGraphFile.getAbsolutePath());
        } catch (RrdException e) {
            logger.error("Error while creating RRD file of "
                    + getName(), e);
        } catch (IOException e) {
            logger.error("Error while creating RRD file of "
                    + getName(), e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.videolan.vlma.common.IVlServer#checkVLC()
     */
    public boolean checkVLC() {
        return true;
    }

    /**
     * Compares two servers.
     * 
     * @param o the object to compare to
     * @return true if objects are the same, false otherwise
     */
    public boolean equals(Object o) {
        if (o.getClass().equals(VlServer.class)) {
            return ((VlServer) o).name.equals(this.name);
        } else {
            return false;
        }
    }

    /**
     * Gives the server hashcode.
     * 
     * @return the server hashcode
     */
    public int hashCode() {
        return name.hashCode();
    }

    /**
     * Gives the server ID (its hashcode in this case)
     * 
     * @return the server ID
     */
    public int getId() {
        return hashCode();
    }
}
