/*
 * Copyright (c) Citrix Systems, Inc.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 *   1) Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 * 
 *   2) Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials
 *      provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Collections;
using System.Collections.Generic;

using CookComputing.XmlRpc;


namespace XenAPI
{
    public partial class Task : XenObject<Task>
    {
        public Task()
        {
        }

        public Task(string uuid,
            string name_label,
            string name_description,
            List<task_allowed_operations> allowed_operations,
            Dictionary<string, task_allowed_operations> current_operations,
            DateTime created,
            DateTime finished,
            task_status_type status,
            XenRef<Host> resident_on,
            double progress,
            string type,
            string result,
            string[] error_info,
            Dictionary<string, string> other_config,
            XenRef<Task> subtask_of,
            List<XenRef<Task>> subtasks)
        {
            this.uuid = uuid;
            this.name_label = name_label;
            this.name_description = name_description;
            this.allowed_operations = allowed_operations;
            this.current_operations = current_operations;
            this.created = created;
            this.finished = finished;
            this.status = status;
            this.resident_on = resident_on;
            this.progress = progress;
            this.type = type;
            this.result = result;
            this.error_info = error_info;
            this.other_config = other_config;
            this.subtask_of = subtask_of;
            this.subtasks = subtasks;
        }

        /// <summary>
        /// Creates a new Task from a Proxy_Task.
        /// </summary>
        /// <param name="proxy"></param>
        public Task(Proxy_Task proxy)
        {
            this.UpdateFromProxy(proxy);
        }

        public override void UpdateFrom(Task update)
        {
            uuid = update.uuid;
            name_label = update.name_label;
            name_description = update.name_description;
            allowed_operations = update.allowed_operations;
            current_operations = update.current_operations;
            created = update.created;
            finished = update.finished;
            status = update.status;
            resident_on = update.resident_on;
            progress = update.progress;
            type = update.type;
            result = update.result;
            error_info = update.error_info;
            other_config = update.other_config;
            subtask_of = update.subtask_of;
            subtasks = update.subtasks;
        }

        internal void UpdateFromProxy(Proxy_Task proxy)
        {
            uuid = proxy.uuid == null ? null : (string)proxy.uuid;
            name_label = proxy.name_label == null ? null : (string)proxy.name_label;
            name_description = proxy.name_description == null ? null : (string)proxy.name_description;
            allowed_operations = proxy.allowed_operations == null ? null : Helper.StringArrayToEnumList<task_allowed_operations>(proxy.allowed_operations);
            current_operations = proxy.current_operations == null ? null : Maps.convert_from_proxy_string_task_allowed_operations(proxy.current_operations);
            created = proxy.created;
            finished = proxy.finished;
            status = proxy.status == null ? (task_status_type) 0 : (task_status_type)Helper.EnumParseDefault(typeof(task_status_type), (string)proxy.status);
            resident_on = proxy.resident_on == null ? null : XenRef<Host>.Create(proxy.resident_on);
            progress = Convert.ToDouble(proxy.progress);
            type = proxy.type == null ? null : (string)proxy.type;
            result = proxy.result == null ? null : (string)proxy.result;
            error_info = proxy.error_info == null ? new string[] {} : (string [])proxy.error_info;
            other_config = proxy.other_config == null ? null : Maps.convert_from_proxy_string_string(proxy.other_config);
            subtask_of = proxy.subtask_of == null ? null : XenRef<Task>.Create(proxy.subtask_of);
            subtasks = proxy.subtasks == null ? null : XenRef<Task>.Create(proxy.subtasks);
        }

        public Proxy_Task ToProxy()
        {
            Proxy_Task result_ = new Proxy_Task();
            result_.uuid = (uuid != null) ? uuid : "";
            result_.name_label = (name_label != null) ? name_label : "";
            result_.name_description = (name_description != null) ? name_description : "";
            result_.allowed_operations = (allowed_operations != null) ? Helper.ObjectListToStringArray(allowed_operations) : new string[] {};
            result_.current_operations = Maps.convert_to_proxy_string_task_allowed_operations(current_operations);
            result_.created = created;
            result_.finished = finished;
            result_.status = task_status_type_helper.ToString(status);
            result_.resident_on = (resident_on != null) ? resident_on : "";
            result_.progress = progress;
            result_.type = (type != null) ? type : "";
            result_.result = (result != null) ? result : "";
            result_.error_info = error_info;
            result_.other_config = Maps.convert_to_proxy_string_string(other_config);
            result_.subtask_of = (subtask_of != null) ? subtask_of : "";
            result_.subtasks = (subtasks != null) ? Helper.RefListToStringArray(subtasks) : new string[] {};
            return result_;
        }

        /// <summary>
        /// Creates a new Task from a Hashtable.
        /// </summary>
        /// <param name="table"></param>
        public Task(Hashtable table)
        {
            uuid = Marshalling.ParseString(table, "uuid");
            name_label = Marshalling.ParseString(table, "name_label");
            name_description = Marshalling.ParseString(table, "name_description");
            allowed_operations = Helper.StringArrayToEnumList<task_allowed_operations>(Marshalling.ParseStringArray(table, "allowed_operations"));
            current_operations = Maps.convert_from_proxy_string_task_allowed_operations(Marshalling.ParseHashTable(table, "current_operations"));
            created = Marshalling.ParseDateTime(table, "created");
            finished = Marshalling.ParseDateTime(table, "finished");
            status = (task_status_type)Helper.EnumParseDefault(typeof(task_status_type), Marshalling.ParseString(table, "status"));
            resident_on = Marshalling.ParseRef<Host>(table, "resident_on");
            progress = Marshalling.ParseDouble(table, "progress");
            type = Marshalling.ParseString(table, "type");
            result = Marshalling.ParseString(table, "result");
            error_info = Marshalling.ParseStringArray(table, "error_info");
            other_config = Maps.convert_from_proxy_string_string(Marshalling.ParseHashTable(table, "other_config"));
            subtask_of = Marshalling.ParseRef<Task>(table, "subtask_of");
            subtasks = Marshalling.ParseSetRef<Task>(table, "subtasks");
        }

        public bool DeepEquals(Task other, bool ignoreCurrentOperations)
        {
            if (ReferenceEquals(null, other))
                return false;
            if (ReferenceEquals(this, other))
                return true;

            if (!ignoreCurrentOperations && !Helper.AreEqual2(this.current_operations, other.current_operations))
                return false;

            return Helper.AreEqual2(this._uuid, other._uuid) &&
                Helper.AreEqual2(this._name_label, other._name_label) &&
                Helper.AreEqual2(this._name_description, other._name_description) &&
                Helper.AreEqual2(this._allowed_operations, other._allowed_operations) &&
                Helper.AreEqual2(this._created, other._created) &&
                Helper.AreEqual2(this._finished, other._finished) &&
                Helper.AreEqual2(this._status, other._status) &&
                Helper.AreEqual2(this._resident_on, other._resident_on) &&
                Helper.AreEqual2(this._progress, other._progress) &&
                Helper.AreEqual2(this._type, other._type) &&
                Helper.AreEqual2(this._result, other._result) &&
                Helper.AreEqual2(this._error_info, other._error_info) &&
                Helper.AreEqual2(this._other_config, other._other_config) &&
                Helper.AreEqual2(this._subtask_of, other._subtask_of) &&
                Helper.AreEqual2(this._subtasks, other._subtasks);
        }

        public override string SaveChanges(Session session, string opaqueRef, Task server)
        {
            if (opaqueRef == null)
            {
                System.Diagnostics.Debug.Assert(false, "Cannot create instances of this type on the server");
                return "";
            }
            else
            {
                if (!Helper.AreEqual2(_other_config, server._other_config))
                {
                    Task.set_other_config(session, opaqueRef, _other_config);
                }

                return null;
            }
        }

        public static Task get_record(Session session, string _task)
        {
            return new Task((Proxy_Task)session.proxy.task_get_record(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static XenRef<Task> get_by_uuid(Session session, string _uuid)
        {
            return XenRef<Task>.Create(session.proxy.task_get_by_uuid(session.uuid, (_uuid != null) ? _uuid : "").parse());
        }

        public static List<XenRef<Task>> get_by_name_label(Session session, string _label)
        {
            return XenRef<Task>.Create(session.proxy.task_get_by_name_label(session.uuid, (_label != null) ? _label : "").parse());
        }

        public static string get_uuid(Session session, string _task)
        {
            return (string)session.proxy.task_get_uuid(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static string get_name_label(Session session, string _task)
        {
            return (string)session.proxy.task_get_name_label(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static string get_name_description(Session session, string _task)
        {
            return (string)session.proxy.task_get_name_description(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static List<task_allowed_operations> get_allowed_operations(Session session, string _task)
        {
            return Helper.StringArrayToEnumList<task_allowed_operations>(session.proxy.task_get_allowed_operations(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static Dictionary<string, task_allowed_operations> get_current_operations(Session session, string _task)
        {
            return Maps.convert_from_proxy_string_task_allowed_operations(session.proxy.task_get_current_operations(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static DateTime get_created(Session session, string _task)
        {
            return session.proxy.task_get_created(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static DateTime get_finished(Session session, string _task)
        {
            return session.proxy.task_get_finished(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static task_status_type get_status(Session session, string _task)
        {
            return (task_status_type)Helper.EnumParseDefault(typeof(task_status_type), (string)session.proxy.task_get_status(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static XenRef<Host> get_resident_on(Session session, string _task)
        {
            return XenRef<Host>.Create(session.proxy.task_get_resident_on(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static double get_progress(Session session, string _task)
        {
            return Convert.ToDouble(session.proxy.task_get_progress(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static string get_type(Session session, string _task)
        {
            return (string)session.proxy.task_get_type(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static string get_result(Session session, string _task)
        {
            return (string)session.proxy.task_get_result(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static string[] get_error_info(Session session, string _task)
        {
            return (string [])session.proxy.task_get_error_info(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static Dictionary<string, string> get_other_config(Session session, string _task)
        {
            return Maps.convert_from_proxy_string_string(session.proxy.task_get_other_config(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static XenRef<Task> get_subtask_of(Session session, string _task)
        {
            return XenRef<Task>.Create(session.proxy.task_get_subtask_of(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static List<XenRef<Task>> get_subtasks(Session session, string _task)
        {
            return XenRef<Task>.Create(session.proxy.task_get_subtasks(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static void set_other_config(Session session, string _task, Dictionary<string, string> _other_config)
        {
            session.proxy.task_set_other_config(session.uuid, (_task != null) ? _task : "", Maps.convert_to_proxy_string_string(_other_config)).parse();
        }

        public static void add_to_other_config(Session session, string _task, string _key, string _value)
        {
            session.proxy.task_add_to_other_config(session.uuid, (_task != null) ? _task : "", (_key != null) ? _key : "", (_value != null) ? _value : "").parse();
        }

        public static void remove_from_other_config(Session session, string _task, string _key)
        {
            session.proxy.task_remove_from_other_config(session.uuid, (_task != null) ? _task : "", (_key != null) ? _key : "").parse();
        }

        public static XenRef<Task> create(Session session, string _label, string _description)
        {
            return XenRef<Task>.Create(session.proxy.task_create(session.uuid, (_label != null) ? _label : "", (_description != null) ? _description : "").parse());
        }

        public static void destroy(Session session, string _self)
        {
            session.proxy.task_destroy(session.uuid, (_self != null) ? _self : "").parse();
        }

        public static void cancel(Session session, string _task)
        {
            session.proxy.task_cancel(session.uuid, (_task != null) ? _task : "").parse();
        }

        public static XenRef<Task> async_cancel(Session session, string _task)
        {
            return XenRef<Task>.Create(session.proxy.async_task_cancel(session.uuid, (_task != null) ? _task : "").parse());
        }

        public static List<XenRef<Task>> get_all(Session session)
        {
            return XenRef<Task>.Create(session.proxy.task_get_all(session.uuid).parse());
        }

        public static Dictionary<XenRef<Task>, Task> get_all_records(Session session)
        {
            return XenRef<Task>.Create<Proxy_Task>(session.proxy.task_get_all_records(session.uuid).parse());
        }

        private string _uuid;
        public virtual string uuid {
             get { return _uuid; }
             set { if (!Helper.AreEqual(value, _uuid)) { _uuid = value; Changed = true; NotifyPropertyChanged("uuid"); } }
         }

        private string _name_label;
        public virtual string name_label {
             get { return _name_label; }
             set { if (!Helper.AreEqual(value, _name_label)) { _name_label = value; Changed = true; NotifyPropertyChanged("name_label"); } }
         }

        private string _name_description;
        public virtual string name_description {
             get { return _name_description; }
             set { if (!Helper.AreEqual(value, _name_description)) { _name_description = value; Changed = true; NotifyPropertyChanged("name_description"); } }
         }

        private List<task_allowed_operations> _allowed_operations;
        public virtual List<task_allowed_operations> allowed_operations {
             get { return _allowed_operations; }
             set { if (!Helper.AreEqual(value, _allowed_operations)) { _allowed_operations = value; Changed = true; NotifyPropertyChanged("allowed_operations"); } }
         }

        private Dictionary<string, task_allowed_operations> _current_operations;
        public virtual Dictionary<string, task_allowed_operations> current_operations {
             get { return _current_operations; }
             set { if (!Helper.AreEqual(value, _current_operations)) { _current_operations = value; Changed = true; NotifyPropertyChanged("current_operations"); } }
         }

        private DateTime _created;
        public virtual DateTime created {
             get { return _created; }
             set { if (!Helper.AreEqual(value, _created)) { _created = value; Changed = true; NotifyPropertyChanged("created"); } }
         }

        private DateTime _finished;
        public virtual DateTime finished {
             get { return _finished; }
             set { if (!Helper.AreEqual(value, _finished)) { _finished = value; Changed = true; NotifyPropertyChanged("finished"); } }
         }

        private task_status_type _status;
        public virtual task_status_type status {
             get { return _status; }
             set { if (!Helper.AreEqual(value, _status)) { _status = value; Changed = true; NotifyPropertyChanged("status"); } }
         }

        private XenRef<Host> _resident_on;
        public virtual XenRef<Host> resident_on {
             get { return _resident_on; }
             set { if (!Helper.AreEqual(value, _resident_on)) { _resident_on = value; Changed = true; NotifyPropertyChanged("resident_on"); } }
         }

        private double _progress;
        public virtual double progress {
             get { return _progress; }
             set { if (!Helper.AreEqual(value, _progress)) { _progress = value; Changed = true; NotifyPropertyChanged("progress"); } }
         }

        private string _type;
        public virtual string type {
             get { return _type; }
             set { if (!Helper.AreEqual(value, _type)) { _type = value; Changed = true; NotifyPropertyChanged("type"); } }
         }

        private string _result;
        public virtual string result {
             get { return _result; }
             set { if (!Helper.AreEqual(value, _result)) { _result = value; Changed = true; NotifyPropertyChanged("result"); } }
         }

        private string[] _error_info;
        public virtual string[] error_info {
             get { return _error_info; }
             set { if (!Helper.AreEqual(value, _error_info)) { _error_info = value; Changed = true; NotifyPropertyChanged("error_info"); } }
         }

        private Dictionary<string, string> _other_config;
        public virtual Dictionary<string, string> other_config {
             get { return _other_config; }
             set { if (!Helper.AreEqual(value, _other_config)) { _other_config = value; Changed = true; NotifyPropertyChanged("other_config"); } }
         }

        private XenRef<Task> _subtask_of;
        public virtual XenRef<Task> subtask_of {
             get { return _subtask_of; }
             set { if (!Helper.AreEqual(value, _subtask_of)) { _subtask_of = value; Changed = true; NotifyPropertyChanged("subtask_of"); } }
         }

        private List<XenRef<Task>> _subtasks;
        public virtual List<XenRef<Task>> subtasks {
             get { return _subtasks; }
             set { if (!Helper.AreEqual(value, _subtasks)) { _subtasks = value; Changed = true; NotifyPropertyChanged("subtasks"); } }
         }


    }
}
