package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"os"
	"path/filepath"
	"runtime"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestSimple(t *testing.T) {
	// Just run a few really simple queries and make sure they work
	// properly.

	cases := []struct {
		input  []byte
		output []byte
		query  string
	}{
		{
			input:  []byte(`{"foo": "bar", "baz": "quux"}`),
			query:  "input.foo",
			output: []byte(`"bar"`),
		},
		{
			input:  []byte(`{"foo": "bar", "baz": "quux"}`),
			query:  "input.foo; input.baz",
			output: []byte(`["bar", "quux"]`),
		},
		{
			input:  []byte(`{"foo": "bar", "baz": "quux"}`),
			query:  "",
			output: []byte(`{"foo": "bar", "baz": "quux"}`),
		},
		{
			input:  []byte(`{"foo": "bar", "baz": "quux"}`),
			query:  "   ",
			output: []byte(`{"foo": "bar", "baz": "quux"}`),
		},
	}

	for i, c := range cases {
		common := &Common{
			InputFormat: "yaml",
			NoColor:     true,
		}

		qc := &QueryCmd{
			Common: common,
			Query:  c.query,
		}

		globals := &Globals{}

		ireader := &bytes.Buffer{}
		ireader.Write(c.input)
		owriter := &bytes.Buffer{}

		stdin = ireader
		stdout = owriter

		err := qc.Run(globals)
		assert.Nil(t, err)

		t.Logf("---- case %d", i)
		t.Logf("expected output: %s\n", string(c.output))
		t.Logf("actual output: %s\n", string(owriter.String()))

		var expected interface{}
		var actual interface{}

		err = json.Unmarshal(owriter.Bytes(), &actual)
		assert.Nil(t, err)

		err = json.Unmarshal(c.output, &expected)
		assert.Nil(t, err)

		assert.Equal(t, expected, actual)
	}
}

func TestPatchIn(t *testing.T) {
	obj := map[string]interface{}{
		"test1": map[string]interface{}{
			"test1/1": "test1/2",
		},
		"test2": "test2/1",
	}

	obj = patchIn([]string{"test1", "test1/1", "foo"}, obj, 123)

	assert.Equal(t, map[string]interface{}{
		"test1": map[string]interface{}{
			"test1/1": map[string]interface{}{
				"foo": 123,
			},
		},
		"test2": "test2/1",
	},
		obj,
	)
}

func getRqPath(t *testing.T) string {
	_, filename, _, ok := runtime.Caller(0)
	if !ok {
		t.Fatal()
	}
	cmdDir := filepath.Dir(filename)
	rqDir := filepath.Dir(filepath.Dir(cmdDir))

	return rqDir
}

func getBooksTOMLPath(t *testing.T) string {
	booksPath := filepath.Join(getRqPath(t), "sample_data", "books.toml")
	return booksPath
}

func getBooksJSONPath(t *testing.T) string {
	booksPath := filepath.Join(getRqPath(t), "sample_data", "books.json")
	return booksPath
}

func getTestPath(t *testing.T) string {
	testPath := filepath.Join(getRqPath(t), "test")
	return testPath
}

func TestLoadData(t *testing.T) {
	booksPath := getBooksJSONPath(t)
	if _, err := os.Stat(booksPath); err != nil {
		t.Logf("err: %v", err)
		t.Fatal()
	}

	data, err := loadData([]string{booksPath}, &Common{})
	assert.Nil(t, err)

	expected := map[string]interface{}{
		"books": []interface{}{
			map[string]interface{}{
				"authors": []interface{}{
					"Thorsten Ball",
				},
				"isbn":  "978-3982016115",
				"title": "Writing An Interpreter In Go",
				"year":  json.Number("2018"),
			},
			map[string]interface{}{
				"authors": []interface{}{
					"Thorsten Ball",
				},
				"isbn":  "978-3982016108",
				"title": "Writing A Compiler In Go",
				"year":  json.Number("2018"),
			},
			map[string]interface{}{
				"authors": []interface{}{
					"Alan A. A. Donovan",
					"Brian W. Kernighan",
				},
				"isbn":  "978-0134190440",
				"title": "The Go Programming Language",
				"year":  json.Number("2015"),
			},
			map[string]interface{}{
				"authors": []interface{}{
					"Andrew Williams",
				},
				"isbn":  "978-1789138412",
				"title": "Hands-On GUI Application Development in Go",
				"year":  json.Number("2019"),
			},
			map[string]interface{}{
				"authors": []interface{}{
					"Andrew Williams",
				},
				"isbn":  "1800563167",
				"title": "Building Cross-Platform GUI Applications with Fyne",
				"year":  json.Number("2021"),
			},
		},
	}

	assert.Equal(t, expected, data)
}

func TestDataPaths1(t *testing.T) {
	// Make sure that --data is respected.

	booksPath := getBooksJSONPath(t)

	if _, err := os.Stat(booksPath); err != nil {
		t.Logf("err: %v", err)
		t.Fatal()
	}

	common := &Common{
		InputFormat: "json",
		NoColor:     true,
		DataPaths:   []string{booksPath},
	}

	c := &QueryCmd{
		Common: common,
		Query:  `{b.title | b := data.books[_]; b.isbn == "1800563167"}`,
	}

	globals := &Globals{}

	ireader := &bytes.Buffer{}
	ireader.Write([]byte("{}"))
	owriter := &bytes.Buffer{}

	stdin = ireader
	stdout = owriter

	err := c.Run(globals)
	assert.Nil(t, err)

	var expected interface{}
	var actual interface{}

	err = json.Unmarshal(owriter.Bytes(), &actual)
	assert.Nil(t, err)

	err = json.Unmarshal([]byte(`["Building Cross-Platform GUI Applications with Fyne"]`), &expected)
	assert.Nil(t, err)

	assert.Equal(t, expected, actual)
}

func TestDataPaths2(t *testing.T) {
	// Make sure that --data is respected.

	booksJSONPath := getBooksJSONPath(t)
	booksTOMLPath := getBooksTOMLPath(t)

	if _, err := os.Stat(booksJSONPath); err != nil {
		t.Logf("err: %v", err)
		t.Fatal()
	}

	if _, err := os.Stat(booksTOMLPath); err != nil {
		t.Logf("err: %v", err)
		t.Fatal()
	}

	common := &Common{
		InputFormat: "json",
		NoColor:     true,
		DataPaths:   []string{booksJSONPath, fmt.Sprintf("toml:rego.path=alternate:%s", booksTOMLPath)},
	}

	c := &QueryCmd{
		// remember data.books is the JSON one, and data.alternate is
		// the TOML one. They come out structured a little different --
		// the JSON has a top level array while the TOML shows up as a
		// dict.
		//
		// The count should come up to be the number of books.
		Query: `count({b | b := data.books[_]; data.alternate[b.isbn]})`,

		Common: common,
	}

	globals := &Globals{}

	ireader := &bytes.Buffer{}
	ireader.Write([]byte("{}"))
	owriter := &bytes.Buffer{}

	stdin = ireader
	stdout = owriter

	err := c.Run(globals)
	assert.Nil(t, err)

	var expected interface{}
	var actual interface{}

	err = json.Unmarshal(owriter.Bytes(), &actual)
	assert.Nil(t, err)

	err = json.Unmarshal([]byte(`5`), &expected)
	assert.Nil(t, err)

	assert.Equal(t, expected, actual)
}

func TestInput(t *testing.T) {
	common := &Common{
		InputFormat: "json", // should be superceded
		Input:       `{"format":"base64", "options": {"base64.data": "eyJoaSI6ICJ0aGVyZSJ9Cg=="}}`,
	}

	c := &QueryCmd{Common: common}

	globals := &Globals{}

	ireader := &bytes.Buffer{}
	ireader.Write([]byte("{}"))
	owriter := &bytes.Buffer{}

	stdin = ireader
	stdout = owriter

	err := c.Run(globals)
	assert.Nil(t, err)

	var expected interface{}
	var actual interface{}

	err = json.Unmarshal(owriter.Bytes(), &actual)
	assert.Nil(t, err)

	err = json.Unmarshal([]byte(`{"hi": "there"}`), &expected)
	assert.Nil(t, err)

	assert.Equal(t, expected, actual)
}

func TestInputFromFile(t *testing.T) {
	common := &Common{
		Input:        getBooksJSONPath(t),
		OutputFormat: "raw",
	}

	c := &QueryCmd{
		Common: common,
		Query:  `{b.isbn | b := input[_]; b.title == "Writing An Interpreter In Go"}`,
	}

	globals := &Globals{}

	ireader := &bytes.Buffer{}
	ireader.Write([]byte("{}"))
	owriter := &bytes.Buffer{}

	stdin = ireader
	stdout = owriter

	err := c.Run(globals)
	assert.Nil(t, err)

	assert.Equal(t, "978-3982016115\n", string(owriter.Bytes()))
}

func TestScript(t *testing.T) {
	// Just run a few really simple queries and make sure they work
	// properly.

	cases := []struct {
		path   string
		input  []byte
		output []byte
	}{
		{
			path:   "script1/script.rego",
			input:  []byte(`{}`),
			output: []byte(`{"x": 78}`),
		},
		{
			path:   "script2/script.rego",
			input:  []byte(`{}`),
			output: []byte(`[{"bar": "hi", "baz": true, "foo": 123}]`),
		},
		{
			path:   "script3/script.rego",
			input:  []byte(`{}`),
			output: []byte(`{"script": {"foobar": "baz"}}`),
		},
	}

	for i, c := range cases {
		t.Logf("---- case %d", i)

		common := &Common{
			NoColor: true,
		}

		sc := &ScriptCmd{
			Common: common,
			File:   filepath.Join(getTestPath(t), c.path),
		}

		globals := &Globals{}

		ireader := &bytes.Buffer{}
		ireader.Write(c.input)
		owriter := &bytes.Buffer{}

		stdin = ireader
		stdout = owriter

		err := sc.Run(globals)
		if err != nil {
			t.Logf("sc.Run err: %s\n", err.Error())
		}
		assert.Nil(t, err)

		t.Logf("expected output: %s\n", string(c.output))
		t.Logf("actual output: %s\n", string(owriter.String()))

		var expected interface{}
		var actual interface{}

		err = json.Unmarshal(owriter.Bytes(), &actual)
		assert.Nil(t, err)

		err = json.Unmarshal(c.output, &expected)
		assert.Nil(t, err)

		assert.Equal(t, expected, actual)
	}
}

func TestListFormats(t *testing.T) {
	expect := `Input Formats:
	awk
	base64
	csv
	dhall
	dotenv
	hcl
	hjson
	ini
	json
	jsonc
	lines
	ndjson
	raw
	tabular
	tf
	toml
	tsv
	xml
	yaml

Output Formats:
	csv
	dotenv
	hcl
	jcs
	json
	md-table
	ndjson
	null
	raw
	sh
	template
	toml
	tsv
	xml
	yaml

NOTE: the 'raw' output format is the same as 'json', except syntax highlighting
and pretty printing are disabled, and if the result is a string, it will be
printed without quotes. If the output is a list of primitive types, they will
be printed one per line without quotes. This can be useful when using rq as a
filter for other shell tools.
`

	owriter := &bytes.Buffer{}
	stdout = owriter

	err := (&ListFormatsCmd{}).Run(&Globals{})
	assert.Nil(t, err)

	assert.Equal(t, expect, owriter.String())
}
