package dotnet

import (
	"strings"

	"github.com/vulncheck-oss/go-exploit/output"
	"github.com/vulncheck-oss/go-exploit/transform"
)

// Primitives are a data type that can be one of a few different types.
// This implemented has been implementede on the ones that we have used so far.
// These are generally added as additionalinfo when there is a BinaryTypeEnum value passed
// as a member type for Primitive.
type Primitive interface {
	PrimToString() string
}

type (
	PrimitiveInt16      int
	PrimitiveInt32      int
	PrimitiveByte       byte
	PrimitiveByteString string
)

func (me PrimitiveInt16) PrimToString() string {
	return transform.PackLittleInt16(int(me))
}

// A placeholder for lesser-used objects such as Single
// Whatever you give it, will be placed in the stream exactly as given
// Can't just pass a string because it will get 'processed' as a lengthPrefixedString, this avoids that.
func (me PrimitiveByteString) PrimToString() string {
	return string(me)
}

func (me PrimitiveInt32) PrimToString() string {
	return transform.PackLittleInt32(int(me))
}

func (me PrimitiveByte) PrimToString() string {
	b := []byte{byte(me)}

	return string(b)
}

// Serialized objects are basically classes that are defined by a series of RecordTypes.
// All existing record types are defined here though all are not used for our purposes.
// ref https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-nrbf/954a0657-b901-4813-9398-4ec732fe8b32
var RecordTypeEnumMap = map[string]int{
	"SerializedStreamHeader":         0,
	"ClassWithId":                    1,
	"SystemClassWithMembers":         2,
	"ClassWithMembers":               3,
	"SystemClassWithMembersAndTypes": 4,
	"ClassWithMembersAndTypes":       5,
	"BinaryObjectString":             6,
	"BinaryArray":                    7,
	"MemberPrimitiveTyped":           8,
	"MemberReference":                9,
	"ObjectNull":                     10,
	"MessageEnd":                     11,
	"BinaryLibrary":                  12,
	"ObjectNullMultiple256":          13,
	"ObjectNullMultiple":             14,
	"ArraySinglePrimitive":           15,
	"ArraySingleObject":              16,
	"ArraySingleString":              17,
	"MethodCall":                     21,
	"MethodReturn":                   22,
}

// Binary type information that is used to define the type of each member of the class being defined.
var BinaryTypeEnumerationMap = map[string]int{
	"Primitive":      0, // Needs Add Info
	"String":         1,
	"Object":         2,
	"SystemClass":    3, // Needs Add Info
	"Class":          4, // Needs Add Info
	"ObjectArray":    5,
	"StringArray":    6,
	"PrimitiveArray": 7, // Needs Add Info
}

// The Primitive Type, must be added to additionalInfo array for each primitive class member that was defined in MemberTypes for a given object.
// https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-nrbf/4e77849f-89e3-49db-8fb9-e77ee4bc7214
var PrimitiveTypeEnum = map[string]int{
	"Boolean": 1,
	"Byte":    2,
	"Char":    3,
	// there is no 4 per https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nrbf/4e77849f-89e3-49db-8fb9-e77ee4bc7214
	"Decimal":  5,
	"Double":   6,
	"Int16":    7,
	"Int32":    8,
	"Int64":    9,
	"SByte":    10,
	"Single":   11,
	"TimeSpan": 12,
	"DateTime": 13,
	"UInt16":   14,
	"UInt32":   15,
	"UInt64":   16,
	"Null":     17,
	"String":   18,
}

// Contains metadata about a class, used for ClassWithMembersAndTypesRecords and SystemClassWithMembersAndTypesRecords.
type ClassInfo struct {
	ObjectID int
	// Needs to be length-prefixed when used
	Name string
	// should match len(MemberNames)
	MemberCount int
	// Exactly what it sounds like.
	MemberNames []string
}

// Class library metadata, sometimes used as additionalinfo value to define the library a class belongs to.
// This is used when a Class is a membervalue for another class.
type ClassTypeInfo struct {
	TypeName  string
	LibraryID int
}

// Defines the types and additional info about the members themselves. used for ClassWithMembersAndTypesRecords and SystemClassWithMembersAndTypesRecords
// ref: https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nrbf/aa509b5a-620a-4592-a5d8-7e9613e0a03e
type MemberTypeInfo struct {
	BinaryTypeEnums []int
	BinaryTypes     []string // for convenience not part of the 'official' data structure per MSDN
	AdditionalInfos []any
}

// Self-explanatory, checks if given BinaryTypeEnum expects additionalInfo so that the function can retrieve a value from that array.
func needsAdditionalInfo(inType string) bool {
	switch inType {
	case "Class":
		return true
	case "SystemClass":
		return true
	case "Primitive":
		return true
	case "PrimitiveArray":
		return true
	default:
		return false
	}
}

// This is Basically a constructor to build MemberTypeInfo into a binary string as expected by the serialization format.
// This uses a constructor because there is validation we want to perform such as length checking and ensuring the provided types are valid.
// ref: https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nrbf/aa509b5a-620a-4592-a5d8-7e9613e0a03e
func getMemberTypeInfo(memberTypes []string, memberNames []string, additionalInfo []any) (MemberTypeInfo, bool) {
	// NOTE: the members param is just being used here for length validation since it's a separate object from the corresponding ClassInfo
	if len(memberNames) != len(memberTypes) {
		output.PrintFrameworkError("Length mismatch between memberTypes and members in getMemberTypeInfo()")

		return MemberTypeInfo{}, false
	}
	addInfoIndex := 0
	memberTypeInfo := MemberTypeInfo{}
	var additionalInfos []any
	memberTypeInfo.AdditionalInfos = additionalInfos

	// build the binary array string of binarytypeenums, which will basically be of type []byte{type0,type1,typeN}
	for _, memberType := range memberTypes {
		val, ok := BinaryTypeEnumerationMap[memberType] // Ensuring that they're valid types
		if !ok {
			output.PrintfFrameworkError("Failed to build MemberTypeInfo string: Invalid member type provided: %s, names: %q , all types: %s", memberType, memberNames, memberTypes)

			return MemberTypeInfo{}, false
		}

		memberTypeInfo.BinaryTypes = append(memberTypeInfo.BinaryTypes, memberType)
		memberTypeInfo.BinaryTypeEnums = append(memberTypeInfo.BinaryTypeEnums, val)

		if needsAdditionalInfo(memberType) {
			if len(additionalInfo) < addInfoIndex+1 {
				output.PrintfFrameworkError("Failed to build MemberTypeInfo string: Not enough additionalInfo values provided: %s", memberType)

				return MemberTypeInfo{}, false
			}
			addInfo := additionalInfo[addInfoIndex]
			addInfoIndex++
			memberTypeInfo.AdditionalInfos = append(memberTypeInfo.AdditionalInfos, addInfo)
		}
	}

	return memberTypeInfo, true
}

// Gives us the expected expected binary string representation.
// MemberTypeInfo output order: byteTypeEnums[]byte + []AdditionalInfo.
func (memberTypeInfo MemberTypeInfo) ToBin() (string, bool) {
	dataSequence := ""
	// build the array of binarytypeenums
	for _, v := range memberTypeInfo.BinaryTypeEnums {
		dataSequence += string(byte(v))
	}

	for _, addInfo := range memberTypeInfo.AdditionalInfos {
		if addInfo == nil {
			output.PrintFrameworkError("Nil additional info provided")

			return "", false
		}

		typeInt, ok := addInfo.(int) // it seems these are primitive type enum values
		if ok {
			dataSequence += string(byte(typeInt))

			continue
		}

		stringInput, ok := addInfo.(string)
		if ok {
			dataSequence += lengthPrefixedString(stringInput)

			continue
		}

		// handling ClassTypeInfo used for 'Class' type
		classTypeInfo, ok := addInfo.(ClassTypeInfo)
		if ok {
			dataSequence += lengthPrefixedString(classTypeInfo.TypeName)
			dataSequence += transform.PackLittleInt32(classTypeInfo.LibraryID)

			continue
		}
		output.PrintfFrameworkError("Unsupported additional info type provided %q", addInfo)

		return "", false
	}

	return dataSequence, true
}

// returns all but the last item in the class name
// obj.Name = "Microsoft.VisualStudio.Text.Formatting.TextFormattingRunProperties"
// obj.GetLeadingClassName(className) == "Microsoft.VisualStudio.Text.Formatting".
func (classInfo ClassInfo) GetLeadingClassName() string {
	split := strings.Split(classInfo.Name, ".")
	sLen := len(split)
	if sLen < 2 {
		output.PrintfFrameworkWarn("Class name does not contain '.' character, entire value returned for GetLeadingClassName(). Name=%s, len %d", classInfo.Name, sLen)

		return classInfo.Name
	}

	return strings.Join(split[:len(split)-1], ".")
}

// returns only the last item in the class name
// obj.Name = "Microsoft.VisualStudio.Text.Formatting.TextFormattingRunProperties"
// obj.GetLeadingClassName(className) == "TextFormattingRunProperties".
func (classInfo ClassInfo) GetBaseClassName() string {
	split := strings.Split(classInfo.Name, ".")
	sLen := len(split)
	if sLen == 0 {
		output.PrintfFrameworkWarn("Class name does not contain '.' character, entire value returned for GetBaseClassName(). Name=%s, len %d", classInfo.Name, sLen)

		return classInfo.Name
	}

	return split[sLen-1]
}

// Gives us the expected expected binary string representation.
// ref: https://learn.microsoft.com/en-us/openspecs/windows_protocols/ms-nrbf/0a192be0-58a1-41d0-8a54-9c91db0ab7bf
func (classInfo ClassInfo) ToBin() string {
	objectIDString := transform.PackLittleInt32(classInfo.ObjectID)
	memberCountString := transform.PackLittleInt32(len(classInfo.MemberNames))

	memberNamesString := ""
	for _, memberName := range classInfo.MemberNames {
		memberNamesString += lengthPrefixedString(memberName)
	}

	return objectIDString + lengthPrefixedString(classInfo.Name) + memberCountString + memberNamesString
}
